module Fastlane
  module Actions
    class UnityAction < Action
      def self.run(params)
        if params[:executable]
          build_cmd = params[:executable]
        else
          text = File.read("#{params[:project_path]}/ProjectSettings/ProjectVersion.txt")
          unity_version = text[/[0-9].*$/]
          build_cmd = params[:unity_hub] << "/" << unity_version << "/Unity.app/Contents/MacOS/Unity"
        end

        build_cmd << " -projectPath #{params[:project_path]}"

        if params[:batch_mode]
          build_cmd << " -batchmode"
        end

        if params[:quit]
          build_cmd << " -quit"
        end

        if params[:no_graphics]
          build_cmd << " -nographics"
        end

        build_cmd << " -executeMethod #{params[:execute_method]}"
        build_cmd << " -buildTarget #{params[:build_target]}"
        build_cmd << " -logFile #{params[:log_file]}"
        build_cmd << " -quitTimeout #{params[:quit_timeout]}"

        UI.message ""
        UI.message Terminal::Table.new(
          title: "Unity".green,
          headings: ["Option", "Value"],
          rows: params.values,
        )
        UI.message ""

        UI.message "Start running"
        UI.message ""

        sh build_cmd

        UI.success "Completed"
      end

      #####################################################
      # @!group Documentation
      #####################################################

      def self.description
        "Run Unity in batch mode"
      end

      def self.available_options
        [
          FastlaneCore::ConfigItem.new(key: :executable,
                                       env_name: "UNITY_EXECUTABLE_PATH",
                                       description: "Override Unity path",
                                       optional: true,
                                       default_value: nil),
          FastlaneCore::ConfigItem.new(key: :unity_hub,
                                       env_name: "UNITY_HUB_PATH",
                                       description: "Path to Unity Hub Editors dir",
                                       optional: true,
                                       default_value: "/Volumes/Data/Unity"),
          FastlaneCore::ConfigItem.new(key: :project_path,
                                       env_name: "UNITY_PROJECT_PATH",
                                       description: "Path to Unity project",
                                       default_value: "#{Dir.pwd}"),
          FastlaneCore::ConfigItem.new(key: :build_target,
                                       env_name: "UNITY_BUILD_TARGET",
                                       description: "Target to build",
                                       optional: true,
                                       default_value: "iOS"),
          FastlaneCore::ConfigItem.new(key: :execute_method,
                                       env_name: "UNITY_EXECUTE_METHOD",
                                       description: "Method to execute",
                                       default_value: nil),
          FastlaneCore::ConfigItem.new(key: :log_file,
                                       env_name: "UNITY_LOG_FILE",
                                       description: "Log File",
                                       optional: true,
                                       default_value: "-"),
          FastlaneCore::ConfigItem.new(key: :quit_timeout,
                                       env_name: "UNITY_QUIT_TIMEOUT",
                                       description: "Quit Timeout",
                                       optional: true,
                                       default_value: 300),
          FastlaneCore::ConfigItem.new(key: :no_graphics,
                                       env_name: "UNITY_NO_GRAPHICS",
                                       description: "Disable graphics mode",
                                       optional: true,
                                       is_string: false,
                                       default_value: true),
          FastlaneCore::ConfigItem.new(key: :batch_mode,
                                       env_name: "UNITY_BATCH_MODE",
                                       description: "Use batch mode",
                                       optional: true,
                                       is_string: false,
                                       default_value: true),
          FastlaneCore::ConfigItem.new(key: :quit,
                                       env_name: "UNITY_QUIT",
                                       optional: true,
                                       is_string: false,
                                       description: "Close Unity after script execution",
                                       default_value: true),
        ]
      end

      def self.authors
        ["Vadim Vlasov"]
      end

      def self.is_supported?(platform)
        [:ios, :android, :windows, :macos, :webgl].include?(platform)
      end
    end
  end
end
