module Fastlane
  module Actions
    class AppgateUploadAction < Action
      def self.run(params)
        require 'faraday'
        require 'faraday_middleware'

        Helper::AppGateHelper.verify_required_params(params)
        api_key = params[:api_key]
        app_id = params[:app_id]
        file_path = params[:file_path]
        branch = params[:branch]
        include_branch = params[:include_branch]
        skip_waiting_for_release = params[:skip_waiting_for_release]
        version = params[:version]
        build_number = params[:build_number]

        if !include_branch
          branch = ""
        end

        UI.message("Starting release upload...")
        release_data = Helper::AppGateHelper.create_release(
          api_key,
          app_id,
          branch,
          File.basename(file_path),
          File.size(file_path),
          version,
          build_number
        )
        UI.abort_with_message!("Failed to create release") unless release_data

        release_id = release_data[:release_id]
        upload_token = release_data[:upload_token]

        uploaded = Helper::AppGateHelper.upload_file_parallel(release_id, file_path, upload_token)
        UI.abort_with_message!("Failed to upload file") unless uploaded

        UI.message("Finishing release...")
        finished = Helper::AppGateHelper.finish_upload(release_id, upload_token)
        UI.abort_with_message!("Failed to finish upload") unless finished

        if skip_waiting_for_release
          UI.success("Successfully uploaded release! Skipping wait for processing.")
        else
          UI.message("Waiting for release to be ready...")
          release_ready = Helper::AppGateHelper.wait_for_release_to_be_ready(release_id, upload_token)
          UI.abort_with_message!("Failed to wait for release to be ready") unless release_ready
          UI.success("Successfully uploaded and finished release!")
        end
      end

      #####################################################
      # @!group Documentation
      #####################################################

      def self.description
        "Upload build to AppGate"
      end

      def self.available_options
        [
          FastlaneCore::ConfigItem.new(key: :app_id,
                                       env_name: "APPGATE_APP_ID",
                                       description: "AppGate App ID",
                                       optional: false,
                                       default_value: ""),
          FastlaneCore::ConfigItem.new(key: :file_path,
                                       env_name: "APPGATE_FILE_PATH",
                                       description: "Path to your build file (IPA, APK, AAB, DMG, PKG, ZIP, EXE, MSI, MSIX, APPX)",
                                       verify_block: proc do |value|
                                         UI.user_error!("Could not find file at path '#{value}'") unless File.exist?(value)
                                       end),
          FastlaneCore::ConfigItem.new(key: :branch,
                                       env_name: "CI_COMMIT_BRANCH",
                                       description: "Branch from which the app is built",
                                       optional: true,
                                       default_value: ""),
          FastlaneCore::ConfigItem.new(key: :include_branch,
                                       env_name: "APPGATE_INCLUDE_BRANCH",
                                       description: "Flag to include branch in the upload",
                                       optional: true,
                                       default_value: true,
                                       type: Boolean),
          FastlaneCore::ConfigItem.new(key: :api_key,
                                       env_name: "APPGATE_API_KEY",
                                       description: "API key for AppGate",
                                       sensitive: true),
          FastlaneCore::ConfigItem.new(key: :skip_waiting_for_release,
                                       env_name: "APPGATE_SKIP_WAITING_FOR_RELEASE",
                                       description: "Skip waiting for release processing",
                                       optional: true,
                                       is_string: false,
                                       default_value: false),
          FastlaneCore::ConfigItem.new(key: :version,
                                       env_name: "APPGATE_VERSION",
                                       description: "Version string (required for Windows, macOS, WebGL platforms where metadata cannot be auto-extracted)",
                                       optional: true,
                                       default_value: nil),
          FastlaneCore::ConfigItem.new(key: :build_number,
                                       env_name: "APPGATE_BUILD_NUMBER",
                                       description: "Build number (required for Windows, macOS, WebGL platforms where metadata cannot be auto-extracted)",
                                       optional: true,
                                       default_value: nil)
        ]
      end

      def self.authors
        ["Vadim Vlasov"]
      end

      def self.is_supported?(platform)
        [:ios, :android, :mac, :windows].include?(platform)
      end
    end
  end
end
