module Fastlane
  module Actions
    class RestoreCacheAction < Action
      def self.run(params)
        if params[:ignore_cache]
          UI.message("Skipping cache restoration.")
          return
        end

        directory = params[:directory]
        cache_storage_path = params[:cache_storage_path]

        if Dir.exist?(cache_storage_path)
          UI.message("Restoring cache...")

          output = `rsync -a --delete #{cache_storage_path}/ #{directory}/`
          puts output unless output.strip.empty?

          UI.success("Cache restored from #{cache_storage_path}")
        else
          UI.message("Cache directory (#{cache_storage_path}) not found. Skipping cache restoration.")
        end
      end

      def self.description
        "Restores the cache from a specified storage path to a local directory"
      end

      def self.authors
        ["Vadim Vlasov"]
      end

      def self.available_options
        [
          FastlaneCore::ConfigItem.new(key: :directory,
                                       description: "Local directory where the cache will be restored",
                                       optional: true,
                                       default_value: "Library",
                                       type: String),
          FastlaneCore::ConfigItem.new(key: :cache_storage_path,
                                       optional: true,
                                       default_value: "/Volumes/DataSSD/Cache/#{ENV['CI_PROJECT_PATH']}/Library",                               
                                       env_name: "CACHE_STORAGE_PATH",
                                       description: "Storage path from where the cache will be restored",
                                       type: String),
          FastlaneCore::ConfigItem.new(key: :ignore_cache,
                                       env_name: "IGNORE_CACHE",
                                       description: "If set, cache restoration will be skipped",
                                       optional: true,
                                       is_string: false,
                                       default_value: false)
        ]
      end

      def self.is_supported?(platform)
        true
      end
    end
  end
end